'use client';

import { useState, useEffect, useRef } from 'react';
import Image from 'next/image';
import { useTranslations } from 'next-intl';
import LakeTokenHeader from './components/Header';
import LakeTokenFooter from './components/Footer';
import Link from 'next/link';
import CounterAnimation from '@/components/CounterAnimation';
import AnimatedElement from '@/components/AnimatedElement';
import TypingText from './components/TypingText';
import AnimatedImage from '@/components/AnimatedImage';
import AnimatedText from '@/components/AnimatedText';
import './lake-token.css';

function AnimatedPartnersBanner() {
  const scrollRef = useRef<HTMLDivElement>(null);
  const [scroll, setScroll] = useState(0);
  const [isMobile, setIsMobile] = useState(false);
  
  useEffect(() => {
    const checkScreenSize = () => {
      setIsMobile(window.innerWidth < 640);
    };
    checkScreenSize();
    window.addEventListener('resize', checkScreenSize);
    return () => window.removeEventListener('resize', checkScreenSize);
  }, []);
  
  // Размеры изображения (можно подстроить под макет)
  const imgWidth = isMobile ? window.innerWidth * 1.5 : 1724; // адаптивная ширина для мобильных
  const imgHeight = 334; // высота Group 662.svg из Figma

  useEffect(() => {
    let frame: number;
    let start: number | undefined;
    const speed = isMobile ? 20 : 40; // px/sec - медленнее на мобильных
    const animate = (timestamp: number) => {
      if (!start) start = timestamp;
      const elapsed = timestamp - start;
      // Прокрутка по ширине одного изображения
      let newScroll = (elapsed * speed / 1000) % imgWidth;
      setScroll(newScroll);
      frame = requestAnimationFrame(animate);
    };
    frame = requestAnimationFrame(animate);
    return () => cancelAnimationFrame(frame);
  }, []);

  const sectionHeight = isMobile ? '30vw' : imgHeight;
  // На мобильных нужно больше дубликатов для бесшовности
  const duplicateCount = isMobile ? 6 : 2;

  return (
    <div className="relative w-full overflow-hidden" style={{height: sectionHeight, background: '#EFEFEF'}}>
      {/* Левый градиент */}
      <div className="pointer-events-none absolute left-0 top-0 h-full w-16 z-10" style={{background: 'linear-gradient(90deg, #EFEFEF 70%, transparent)'}} />
      {/* Правый градиент */}
      <div className="pointer-events-none absolute right-0 top-0 h-full w-16 z-10" style={{background: 'linear-gradient(270deg, #EFEFEF 70%, transparent)'}} />
      <div
        ref={scrollRef}
        className="flex items-center h-full"
        style={{
          transform: `translateX(-${scroll}px)`,
          willChange: 'transform',
          width: `${imgWidth * duplicateCount}px`,
        }}
      >
        {Array.from({ length: duplicateCount }, (_, index) => (
          <img
            key={index}
            src="/images/lake-token/body/Group 662.svg"
            alt={`Партнёры ${index + 1}`}
            style={{
              width: imgWidth,
              height: '100%',
              maxWidth: 'none',
              display: 'block',
              userSelect: 'none',
              pointerEvents: 'none',
            }}
            draggable={false}
          />
        ))}
      </div>
    </div>
  );
}

function AutoPlayOnVisibleVideo({ src, poster, className, ...props }: { src: string; poster?: string; className?: string; [x: string]: any }) {
  const videoRef = useRef<HTMLVideoElement | null>(null);
  const [isVisible, setIsVisible] = useState(false);

  useEffect(() => {
    const observer = new window.IntersectionObserver(
      ([entry]) => {
        setIsVisible(entry.isIntersecting);
      },
      { threshold: 0.3 }
    );
    if (videoRef.current) observer.observe(videoRef.current);
    return () => {
      if (videoRef.current) observer.unobserve(videoRef.current);
    };
  }, []);

  useEffect(() => {
    if (videoRef.current) {
      if (isVisible) {
        videoRef.current.play();
      } else {
        videoRef.current.pause();
      }
    }
  }, [isVisible]);

  return (
    <video
      ref={videoRef}
      src={src}
      poster={poster}
      muted
      playsInline
      loop
      controls={false}
      className={className}
      style={{ maxWidth: 534, maxHeight: 632, width: '100%', height: '100%', objectFit: 'cover', background: '#000' }}
      {...props}
    />
  );
}

function InvestmentVectorCarousel() {
  const [isMobile, setIsMobile] = useState(false);
  useEffect(() => {
    const checkScreenSize = () => {
      setIsMobile(window.innerWidth < 640);
    };
    checkScreenSize();
    window.addEventListener('resize', checkScreenSize);
    return () => window.removeEventListener('resize', checkScreenSize);
  }, []);
  const fadeWidth = isMobile ? '0.5rem' : '8rem';
  const carouselHeight = isMobile ? 50 : 100;
  const imgStyle = isMobile ? { height: '40px', width: 'auto' } : { height: '80px', width: 'auto' };
  const gap = isMobile ? '2rem' : '3rem';

  return (
    <div
      className={`relative w-full flex flex-col lake-token-carousel${isMobile ? '' : ' items-center py-8'}`}
      style={isMobile ? { padding: 0, alignItems: 'stretch' } : {}}
    >
      {/* Левый fade */}
      <div className="pointer-events-none absolute left-0 top-0 h-full z-10" style={{width: fadeWidth, background: 'linear-gradient(90deg, white 70%, transparent)'}} />
      {/* Правый fade */}
      <div className="pointer-events-none absolute right-0 top-0 h-full z-10" style={{width: fadeWidth, background: 'linear-gradient(270deg, white 70%, transparent)'}} />
      {/* Первая карусель (перемешанные изображения, движение справа налево) */}
      <div className="relative w-full overflow-hidden flex justify-center mb-8" style={{height: carouselHeight}}>
        <div
          className="flex animate-scroll-right-to-left"
          style={{
            width: 2 * (5 * (320 + 32) - 32),
            willChange: 'transform',
            gap: gap,
          }}
        >
          {/* Первый набор изображений (перемешанный) */}
          <Image src="/images/lake-token/body/Group 716.svg" alt="Group 716" width={91.942} height={51.568} style={imgStyle} />
          <Image src="/images/lake-token/body/Mask group1.svg" alt="Mask group1" width={167.596} height={51.568} style={imgStyle} />
          <Image src="/images/lake-token/body/Group 714.svg" alt="Group 714" width={236.362} height={51.568} style={imgStyle} />
          <Image src="/images/lake-token/body/Group 717.svg" alt="Group 717" width={235.989} height={51.568} style={imgStyle} />
          <Image src="/images/lake-token/body/Group 715.svg" alt="Group 715" width={260.939} height={51.568} style={imgStyle} />
          <Image src="/images/lake-token/body/HashKeyGlobal.svg" alt="HashKeyGlobal" width={260.939} height={51.568} style={imgStyle}/>
          {/* Дублированный набор для бесшовности */}
          <Image src="/images/lake-token/body/Group 716.svg" alt="Group 716 duplicate" width={91.942} height={51.568} style={imgStyle} />
          <Image src="/images/lake-token/body/Mask group1.svg" alt="Mask group1 duplicate" width={167.596} height={51.568} style={imgStyle} />
          <Image src="/images/lake-token/body/Group 714.svg" alt="Group 714 duplicate" width={236.362} height={51.568} style={imgStyle} />
          <Image src="/images/lake-token/body/Group 717.svg" alt="Group 717 duplicate" width={235.989} height={51.568} style={imgStyle} />
          <Image src="/images/lake-token/body/Group 715.svg" alt="Group 715 duplicate" width={260.939} height={51.568} style={imgStyle} />
          <Image src="/images/lake-token/body/HashKeyGlobal.svg" alt="HashKeyGlobal" width={260.939} height={51.568} style={imgStyle}/>
        </div>
      </div>
      {/* Вторая и третья карусели — аналогично, с теми же style/gap/imgStyle */}
      <div className="relative w-full overflow-hidden flex justify-center mb-8" style={{height: carouselHeight}}>
        <div
          className="flex animate-scroll-left-to-right"
          style={{
            width: 2 * (5 * (320 + 32) - 32),
            willChange: 'transform',
            gap: gap,
          }}
        >
          <Image src="/images/lake-token/body/Mask group1.svg" alt="Mask group1" width={167.596} height={51.568} style={imgStyle} />
          <Image src="/images/lake-token/body/Group 715.svg" alt="Group 715" width={260.939} height={51.568} style={imgStyle} />
          <Image src="/images/lake-token/body/Group 716.svg" alt="Group 716" width={91.942} height={51.568} style={imgStyle} />
          <Image src="/images/lake-token/body/Group 714.svg" alt="Group 714" width={236.362} height={51.568} style={imgStyle} />
          <Image src="/images/lake-token/body/Group 717.svg" alt="Group 717" width={235.989} height={51.568} style={imgStyle} />
          <Image src="/images/lake-token/body/Mask group1.svg" alt="Mask group1 duplicate" width={167.596} height={51.568} style={imgStyle} />
          <Image src="/images/lake-token/body/Group 715.svg" alt="Group 715 duplicate" width={260.939} height={51.568} style={imgStyle} />
          <Image src="/images/lake-token/body/Group 716.svg" alt="Group 716 duplicate" width={91.942} height={51.568} style={imgStyle} />
          <Image src="/images/lake-token/body/Group 714.svg" alt="Group 714 duplicate" width={236.362} height={51.568} style={imgStyle} />
          <Image src="/images/lake-token/body/Group 717.svg" alt="Group 717 duplicate" width={235.989} height={51.568} style={imgStyle} />
          <Image src="/images/lake-token/body/HashKeyGlobal.svg" alt="HashKeyGlobal" width={260.939} height={51.568} style={imgStyle}/>
        </div>
      </div>
      <div className="relative w-full overflow-hidden flex justify-center" style={{height: carouselHeight}}>
        <div
          className="flex animate-scroll-right-to-left"
          style={{
            width: 2 * (5 * (320 + 32) - 32),
            willChange: 'transform',
            gap: gap,
          }}
        >
          <Image src="/images/lake-token/body/Group 715.svg" alt="Group 715" width={260.939} height={51.568} style={imgStyle} />
          <Image src="/images/lake-token/body/Group 714.svg" alt="Group 714" width={236.362} height={51.568} style={imgStyle} />
          <Image src="/images/lake-token/body/Mask group1.svg" alt="Mask group1" width={167.596} height={51.568} style={imgStyle} />
          <Image src="/images/lake-token/body/Group 716.svg" alt="Group 716" width={91.942} height={51.568} style={imgStyle} />
          <Image src="/images/lake-token/body/Group 717.svg" alt="Group 717" width={235.989} height={51.568} style={imgStyle} />
          <Image src="/images/lake-token/body/Group 715.svg" alt="Group 715 duplicate" width={260.939} height={51.568} style={imgStyle} />
          <Image src="/images/lake-token/body/Group 714.svg" alt="Group 714 duplicate" width={236.362} height={51.568} style={imgStyle} />
          <Image src="/images/lake-token/body/Mask group1.svg" alt="Mask group1 duplicate" width={167.596} height={51.568} style={imgStyle} />
          <Image src="/images/lake-token/body/Group 716.svg" alt="Group 716 duplicate" width={91.942} height={51.568} style={imgStyle} />
          <Image src="/images/lake-token/body/Group 717.svg" alt="Group 717 duplicate" width={235.989} height={51.568} style={imgStyle} />
          <Image src="/images/lake-token/body/HashKeyGlobal.svg" alt="HashKeyGlobal" width={260.939} height={51.568} style={imgStyle}/>
        </div>
      </div>
    </div>
  );
}

// Вспомогательный компонент для выравнивания количества строк
function FixedLinesText({ children, lines = 2, className = '', ...props }: { children: React.ReactNode, lines?: number, className?: string, [key: string]: any }) {
  // Разбиваем текст на строки по <br> или по \n
  let text = typeof children === 'string' ? children : '';
  let arr = text.split(/<br\s*\/?>|\n/);
  if (arr.length < lines) {
    arr = arr.concat(Array(lines - arr.length).fill(''));
  }
  return (
    <div className={className} {...props}>
      {arr.map((line, i) => (
        <div key={i} style={{ minHeight: '0.1em' }}>{line}</div>
      ))}
    </div>
  );
}

export default function LakeTokenPage() {
  const [isDesktop, setIsDesktop] = useState(false);
  const [isSticky, setIsSticky] = useState(false);
  const t = useTranslations('LakeToken');
  const whatIsLakeRef = useRef<HTMLDivElement | null>(null);

  useEffect(() => {
    const checkScreenSize = () => {
      setIsDesktop(window.innerWidth >= 1024);
    };

    checkScreenSize();
    window.addEventListener('resize', checkScreenSize);

    return () => window.removeEventListener('resize', checkScreenSize);
  }, []);

  useEffect(() => {
    if (!whatIsLakeRef.current) return;
    const observer = new window.IntersectionObserver(
      ([entry]) => {
        setIsSticky(entry.isIntersecting);
      },
      { threshold: 0.01 }
    );
    observer.observe(whatIsLakeRef.current);
    return () => observer.disconnect();
  }, []);

  return (
    <div className="w-full min-h-screen bg-[#EFEFEF] flex flex-col overflow-x-hidden">
      {/* Header */}
      <AnimatedElement animation="fadeIn" delay={0}>
        <LakeTokenHeader />
      </AnimatedElement>

      {/* Hero + What is Lake Token sticky logic */}
      <div className="relative w-full" style={{ height: '200vh' }}>
        <div className={isSticky ? 'sticky top-0 left-0 z-0' : ''} style={{ height: '100vh' }}>
          <section id="lake_token" className="w-full h-screen flex items-center py-6 sm:py-8 md:py-20">
            <div className="lake-token-container flex flex-col lg:flex-row items-center justify-between gap-6 sm:gap-8 lg:gap-0 z-auto w-full">
              <div className="flex flex-col gap-2 sm:gap-4 max-w-full lg:max-w-[580px] w-full">
                <h1 className="lake-token-hero-title text-[32px] sm:text-[48px] md:text-[64px] lg:text-[96px] xl:text-[189.43px] leading-[1.1] break-words overflow-hidden">
                  <span style={{ fontFamily: "'Tilt Warp', sans-serif" }} className="text-[#002D63]">LAKE</span>
                  <br/>
                  <TypingText
                    text="Token"
                    className="font-inter text-[#1C1C1C] tracking-[-2px] sm:tracking-[-4px] md:tracking-[-8px] lg:tracking-[-11.366px] font-normal z-auto"
                    speed={90}
                    cursor={true}
                  />
                </h1>
                <p className="text-[#002D64] text-sm sm:text-base md:text-xl lg:text-2xl font-normal leading-normal">
                  {t('subtitle')}
                </p>
              </div>
              <video
                src="/images/lake-token/body/ГЛАВНАЯ_заменить_статику_и_ABOUT_справа.MP4"
                width={320}
                height={320}
                className="object-cover rounded-[24px] sm:rounded-[32px] md:rounded-[50px] w-full max-w-[320px] sm:max-w-[491px] h-auto"
                autoPlay
                muted
                loop
                playsInline
              />
            </div>
          </section>
        </div>
        {/* What is Lake Token Section - Full Width */}
        <AnimatedElement animation="slideInUp" delay={0}>
        <section
          id="what_is_lake"
          ref={whatIsLakeRef}
          className="w-full min-h-[480px] sm:h-[769px] bg-gradient-to-b from-[#114381] to-[#002C62] flex items-center relative"
          style={{ minHeight: '480px' }}
        >
          <div className="absolute right-0 top-0 h-full w-auto max-w-[60vw] hidden md:block">
            <AnimatedImage
              src="/images/lake-token/body/Group 15.svg"
              alt="Decorative circle"
              width={786}
              height={786}
              className="h-full w-auto object-contain"
              animation="fadeIn"
              delay={200}
            />
          </div>
          <div className="w-full relative">
            <div className="lake-token-section-content flex flex-col gap-8 sm:gap-16 px-2 sm:px-4 md:px-[80px] xl:px-[294px]">
              <div className="flex flex-col gap-4 sm:gap-8">
                <div className="w-full flex justify-center items-center">
                  <AnimatedText animation="slideInUp" delay={0}>
                    <h2 className="text-white font-inter text-2xl sm:text-[32px] md:text-[55.067px] font-normal leading-[120%] text-center whitespace-nowrap w-full" style={{maxWidth: '100vw'}}>
                      {t('what_is_title')}
                    </h2>
                  </AnimatedText>
                </div>
              </div>
              <div className="grid grid-cols-1 md:grid-cols-2 gap-6 sm:gap-12">
                <div className="flex flex-col gap-4 sm:gap-8">
                  {/* Feature 1 */}
                  <AnimatedElement animation="slideInUp" delay={0}>
                  <div className="flex gap-3 sm:gap-6 items-start">
                    <div className="w-12 h-12 sm:w-[79px] sm:h-[79px] flex-shrink-0">
                      <AnimatedImage
                        src="/images/lake-token/body/Mask group.svg"
                        alt="Villa"
                        width={79}
                        height={79}
                        className="w-full h-full object-contain"
                        animation="zoomIn"
                        delay={0}
                      />
                    </div>
                    <div>
                      <FixedLinesText lines={2} className="text-white text-base sm:text-lg font-bold mb-2">{t('what_is_lake.exclusive_access_title')}</FixedLinesText>
                      <FixedLinesText lines={3} className="text-white text-sm sm:text-lg font-normal">{t('what_is_lake.exclusive_access_desc')}</FixedLinesText>
                    </div>
                  </div>
                  </AnimatedElement>
                  {/* Feature 2 */}
                  <AnimatedElement animation="slideInUp" delay={100}>
                  <div className="flex gap-3 sm:gap-6 items-start">
                    <div className="w-12 h-12 sm:w-[79px] sm:h-[79px] flex-shrink-0">
                      <AnimatedImage
                        src="/images/lake-token/body/Mask group 2.svg"
                        alt="DAO"
                        width={79}
                        height={79}
                        className="w-full h-full object-contain"
                        animation="zoomIn"
                        delay={100}
                      />
                    </div>
                    <div>
                      <FixedLinesText lines={2} className="text-white text-base sm:text-lg font-bold mb-2">{t('what_is_lake.dao_governance_title')}</FixedLinesText>
                      <FixedLinesText lines={3} className="text-white text-sm sm:text-lg font-normal">{t('what_is_lake.dao_governance_desc')}</FixedLinesText>
                    </div>
                  </div>
                  </AnimatedElement>
                  {/* Feature 3 */}
                  <AnimatedElement animation="slideInUp" delay={200}>
                  <div className="flex gap-3 sm:gap-6 items-start">
                    <div className="w-12 h-12 sm:w-[79px] sm:h-[79px] flex-shrink-0">
                      <AnimatedImage
                        src="/images/lake-token/body/Mask group 3.svg"
                        alt="Rewards"
                        width={79}
                        height={79}
                        className="w-full h-full object-contain"
                        animation="zoomIn"
                        delay={200}
                      />
                    </div>
                    <div>
                      <FixedLinesText lines={2} className="text-white text-base sm:text-lg font-bold mb-2">{t('what_is_lake.loyalty_rewards_title')}</FixedLinesText>
                      <FixedLinesText lines={3} className="text-white text-sm sm:text-lg font-normal">{t('what_is_lake.loyalty_rewards_desc')}</FixedLinesText>
                    </div>
                  </div>
                  </AnimatedElement>
                </div>
                <div className="flex flex-col gap-4 sm:gap-8">
                  {/* Feature 4 */}
                  <AnimatedElement animation="slideInUp" delay={300}>
                  <div className="flex gap-3 sm:gap-6 items-start">
                    <div className="w-12 h-12 sm:w-[79px] sm:h-[79px] flex-shrink-0">
                      <AnimatedImage
                        src="/images/lake-token/body/Mask group 4.svg"
                        alt="Staking"
                        width={79}
                        height={79}
                        className="w-full h-full object-contain"
                        animation="zoomIn"
                        delay={300}
                      />
                    </div>
                    <div>
                      <FixedLinesText lines={2} className="text-white text-base sm:text-lg font-bold mb-2">{t('what_is_lake.staking_yield_title')}</FixedLinesText>
                      <FixedLinesText lines={3} className="text-white text-sm sm:text-lg font-normal">{t('what_is_lake.staking_yield_desc')}</FixedLinesText>
                    </div>
                  </div>
                  </AnimatedElement>
                  {/* Feature 5 */}
                  <AnimatedElement animation="slideInUp" delay={400}>
                  <div className="flex gap-3 sm:gap-6 items-start">
                    <div className="w-12 h-12 sm:w-[79px] sm:h-[79px] flex-shrink-0">
                      <AnimatedImage
                        src="/images/lake-token/body/Mask group 5.svg"
                        alt="Land"
                        width={79}
                        height={79}
                        className="w-full h-full object-contain"
                        animation="zoomIn"
                        delay={400}
                      />
                    </div>
                    <div>
                      <FixedLinesText lines={2} className="text-white text-base sm:text-lg font-bold mb-2">{t('what_is_lake.land_backed_title')}</FixedLinesText>
                      <FixedLinesText lines={3} className="text-white text-sm sm:text-lg font-normal">{t('what_is_lake.land_backed_desc')}</FixedLinesText>
                    </div>
                  </div>
                  </AnimatedElement>
                  {/* Feature 6 */}
                  <AnimatedElement animation="slideInUp" delay={500}>
                  <div className="flex gap-3 sm:gap-6 items-start">
                    <div className="w-12 h-12 sm:w-[79px] sm:h-[79px] flex-shrink-0">
                        <AnimatedImage
                        src="/images/lake-token/body/Group 14.svg"
                          alt="LAKE Logo"
                        width={79}
                        height={79}
                        className="w-full h-full object-contain"
                        animation="zoomIn"
                        delay={500}
                        />
                    </div>
                    <div>
                      <FixedLinesText lines={2} className="text-white text-base sm:text-lg font-bold mb-2">{t('what_is_lake.lake_token_title')}</FixedLinesText>
                      <FixedLinesText lines={3} className="text-white text-sm sm:text-lg font-normal">{t('what_is_lake.lake_token_desc')}</FixedLinesText>
                    </div>
                  </div>
                  </AnimatedElement>
                </div>
              </div>
            </div>
          </div>
        </section>
        </AnimatedElement>
        {/* Why Lake Token Section - Full Width */}
        <AnimatedElement animation="slideInUp" delay={100}>
        <section id="why_lake_stands" className="relative w-full min-h-[300px] sm:min-h-[400px] bg-[#EFEFEF] flex flex-col md:flex-row justify-center items-center py-6 sm:py-12">
          {/* Декоративная картинка справа */}
          <AnimatedImage
            src="/images/lake-token/body/Group 15.svg"
            alt="Decorative circle"
            width={786}
            height={786}
            className="hidden md:block absolute right-0 top-0 h-full max-h-[786px] w-auto z-0 pointer-events-none select-none"
            animation="fadeIn"
            delay={200}
          />
          <div className="w-full h-16 sm:h-36 bg-gradient-to-b from-[rgba(217,217,217,0.5)] to-[rgba(255,255,255,0)] absolute top-0 left-0 z-10" />
          <div className="w-full max-w-2xl sm:max-w-4xl md:max-w-6xl flex flex-col md:flex-row justify-between items-center gap-4 sm:gap-8 md:gap-8 relative z-10 px-2 sm:px-4">
            <div className="w-full max-w-xs sm:max-w-xl aspect-[267/316] flex items-center justify-center px-2 sm:px-4">
              <AnimatedElement animation="zoomIn" delay={200}>
                <AutoPlayOnVisibleVideo src="/images/lake-token/body/Comp_6.mp4" className="w-full h-full object-cover rounded-xl sm:rounded-2xl shadow-lg" poster={undefined} />
              </AnimatedElement>
            </div>
            <div className="flex-1 flex flex-col justify-between max-w-xl px-2 sm:px-4">
              <AnimatedText animation="slideInUp" delay={0}>
                <h2 className="font-inter text-xl sm:text-3xl md:text-5xl leading-tight mb-4 sm:mb-6"
                  dangerouslySetInnerHTML={{
                    __html: String(t.rich('why_lake_title', {
                      span: (chunks) => `<span class=\"text-[#002D64] font-bold\">${chunks}</span>`
                    }))
                  }}
                />
              </AnimatedText>
              <div className="flex flex-col gap-2 sm:gap-6">
                <AnimatedText animation="fadeIn" delay={100}>
                  <p className="font-inter text-sm sm:text-base md:text-lg font-normal leading-normal"
                    dangerouslySetInnerHTML={{
                      __html: String(t.rich('why_lake.real_yield', {
                        span: (chunks) => `<span class=\"text-[#002D64] font-bold\">${chunks}</span>`
                      }))
                    }}
                  />
                </AnimatedText>
                <AnimatedText animation="fadeIn" delay={200}>
                  <p className="font-inter text-sm sm:text-base md:text-lg font-normal leading-normal"
                    dangerouslySetInnerHTML={{
                      __html: String(t.rich('why_lake.expertise', {
                        span: (chunks) => `<span class=\"text-[#002D64] font-bold\">${chunks}</span>`
                      }))
                    }}
                  />
                </AnimatedText>
                <AnimatedText animation="fadeIn" delay={300}>
                  <p className="font-inter text-sm sm:text-base md:text-lg font-normal leading-normal"
                    dangerouslySetInnerHTML={{
                      __html: String(t.rich('why_lake.emerging_market', {
                        span: (chunks) => `<span class=\"text-[#002D64] font-bold\">${chunks}</span>`
                      }))
                    }}
                  />
                </AnimatedText>
                <AnimatedText animation="fadeIn" delay={400}>
                  <p className="font-inter text-sm sm:text-base md:text-lg font-normal leading-normal"
                    dangerouslySetInnerHTML={{
                      __html: String(t.rich('why_lake.scalable_model', {
                        span: (chunks) => `<span class=\"text-[#002D64] font-bold\">${chunks}</span>`
                      }))
                    }}
                  />
                </AnimatedText>
              </div>
            </div>
          </div>
        </section>
        </AnimatedElement>
        {/* Safety Section - Full Width */}
        <AnimatedElement animation="slideInUp" delay={200}>
        <section id="safety" className="relative w-full bg-[#EFEFEF] py-10 sm:py-20 min-h-[300px] sm:min-h-[400px] flex flex-col items-center">
          <div className="w-full h-16 sm:h-[142px] bg-gradient-to-b from-[rgba(217,217,217,0.5)] to-transparent flex-shrink-0"></div>
          <div className="w-full max-w-2xl sm:max-w-4xl md:max-w-[1140px] flex flex-col items-center">
            <AnimatedText animation="slideInUp" delay={0}>
              <h2 className="text-2xl sm:text-[32px] md:text-[55.067px] font-medium leading-[120%] text-center mb-8 sm:mb-[60px] max-w-[869px]"
                dangerouslySetInnerHTML={{
                  __html: String(t.rich('safety_title', {
                    span: (chunks) => `<span class=\"text-[#002D64]\">${chunks}</span>`
                  }))
                }}
              />
            </AnimatedText>
            <div className="w-full flex flex-col gap-8 sm:gap-[70px] mb-8 sm:mb-[70px]">
              {/* Row 1 */}
              <div className="flex flex-col sm:flex-row justify-between items-stretch sm:items-start gap-4 sm:gap-0">
                {/* Robust Security */}
                <AnimatedElement animation="slideInUp" delay={0}>
                  <div className="flex gap-3 sm:gap-6 items-start w-full sm:w-[476px]">
                    <div className="w-12 h-12 sm:w-[79px] sm:h-[79px] flex-shrink-0 flex items-center justify-center">
                      <AnimatedImage
                        src="/images/lake-token/body/Mask group 2-1.svg"
                        alt="Security"
                        width={79}
                        height={79}
                        className="object-contain"
                        animation="zoomIn"
                        delay={0}
                      />
                    </div>
                    <div>
                      <AnimatedText animation="fadeIn" delay={100}>
                        <h3 className="text-[#1C1C1C] text-base sm:text-lg font-bold mb-2">{t('safety.robust_title')}</h3>
                      </AnimatedText>
                      <AnimatedText animation="fadeIn" delay={200}>
                        <p className="text-[#1C1C1C] text-sm sm:text-lg font-normal">
                          {t('safety.robust_desc')}
                        </p>
                      </AnimatedText>
                    </div>
                  </div>
                </AnimatedElement>
                {/* Optimized Development */}
                <AnimatedElement animation="slideInUp" delay={100}>
                  <div className="flex gap-3 sm:gap-6 items-start w-full sm:w-[493px]">
                    <div className="w-12 h-12 sm:w-[79px] sm:h-[79px] flex-shrink-0 flex items-center justify-center">
                      <AnimatedImage
                        src="/images/lake-token/body/Mask group 2-2.svg"
                        alt="Development"
                        width={79}
                        height={79}
                        className="object-contain"
                        animation="zoomIn"
                        delay={100}
                      />
                    </div>
                    <div>
                      <AnimatedText animation="fadeIn" delay={200}>
                        <h3 className="text-[#1C1C1C] text-base sm:text-lg font-bold mb-2">{t('safety.optimized_title')}</h3>
                      </AnimatedText>
                      <AnimatedText animation="fadeIn" delay={300}>
                        <p className="text-[#1C1C1C] text-sm sm:text-lg font-normal">
                          {t('safety.optimized_desc')}
                        </p>
                      </AnimatedText>
                    </div>
                  </div>
                </AnimatedElement>
              </div>
              {/* Row 2 */}
              <div className="flex flex-col sm:flex-row justify-between items-stretch sm:items-start gap-4 sm:gap-0">
                {/* Price Stability */}
                <AnimatedElement animation="slideInUp" delay={200}>
                  <div className="flex gap-3 sm:gap-6 items-start w-full sm:w-[506px]">
                    <div className="w-12 h-12 sm:w-[79px] sm:h-[79px] flex-shrink-0 flex items-center justify-center">
                      <AnimatedImage
                        src="/images/lake-token/body/Mask group 2-3.svg"
                        alt="Stability"
                        width={79}
                        height={79}
                        className="object-contain"
                        animation="zoomIn"
                        delay={200}
                      />
                    </div>
                    <div>
                      <AnimatedText animation="fadeIn" delay={300}>
                        <h3 className="text-[#1C1C1C] text-base sm:text-lg font-bold mb-2">{t('safety.stability_title')}</h3>
                      </AnimatedText>
                      <AnimatedText animation="fadeIn" delay={400}>
                        <p className="text-[#1C1C1C] text-sm sm:text-lg font-normal">
                          {t('safety.stability_desc')}
                        </p>
                      </AnimatedText>
                    </div>
                  </div>
                </AnimatedElement>
                {/* Smart Contract Verification */}
                <AnimatedElement animation="slideInUp" delay={300}>
                  <div className="flex gap-3 sm:gap-6 items-start w-full sm:w-[493px]">
                    <div className="w-12 h-12 sm:w-[79px] sm:h-[79px] flex-shrink-0 flex items-center justify-center">
                      <AnimatedImage
                        src="/images/lake-token/body/Mask group 2-4.svg"
                        alt="Verification"
                        width={79}
                        height={79}
                        className="object-contain"
                        animation="zoomIn"
                        delay={300}
                      />
                    </div>
                    <div>
                      <AnimatedText animation="fadeIn" delay={400}>
                        <h3 className="text-[#1C1C1C] text-base sm:text-lg font-bold mb-2">{t('safety.verification_title')}</h3>
                      </AnimatedText>
                      <AnimatedText animation="fadeIn" delay={500}>
                        <p className="text-[#1C1C1C] text-sm sm:text-lg font-normal">
                          {t('safety.verification_desc')}
                        </p>
                      </AnimatedText>
                    </div>
                  </div>
                </AnimatedElement>
              </div>
            </div>
            <div className="flex justify-center w-full z-20 mt-4 sm:mt-0">
              <AnimatedElement animation="slideInUp" delay={400}>
                <Link href="/lake-token/security"
                  className="block w-full sm:w-[463px] max-w-full sm:max-w-[463px] h-[72px] sm:h-[84px] bg-[#002D64] rounded-[18px] sm:rounded-[15px] flex items-center justify-center px-6"
                  style={{ flexShrink: 0 }}
                >
                  <span className="text-white text-[20px] sm:text-2xl font-bold">{t('safety.more_details_button')}</span>
                </Link>
              </AnimatedElement>
            </div>
          </div>
        </section>
        </AnimatedElement>
        {/* Roadmap Section - Full Width */}
        <AnimatedElement animation="slideInUp" delay={300}>
        <section id="roadmap_highlights" className="relative w-full bg-[#EFEFEF] py-8 sm:py-12 md:py-20">
          <div className="w-full h-8 sm:h-[142px] bg-gradient-to-b from-[rgba(217,217,217,0.5)] to-transparent"></div>
          <div className="w-full px-2 sm:px-4 md:px-[80px] xl:px-[294px]">
            <AnimatedText animation="slideInUp" delay={0}>
              <h2 className="text-[#002D64] text-2xl sm:text-[32px] md:text-[55.067px] font-bold leading-[120%] mb-4 sm:mb-8 text-center">
                {t('roadmap_highlights')}
              </h2>
            </AnimatedText>
            <div className="max-w-2xl sm:max-w-4xl md:max-w-[1140px] mx-auto flex flex-col gap-4 sm:gap-8">
              {/* Заголовки */}
              <AnimatedElement animation="fadeIn" delay={100}>
                <div className="flex flex-row justify-between items-center gap-0 sm:flex-row sm:items-end">
                  <div className="text-[#1C1C1C] text-base sm:text-2xl font-bold font-inter min-w-[120px] text-left">{t('roadmap.period')}</div>
                  <div className="text-[#1C1C1C] text-base sm:text-2xl font-bold font-inter max-w-[800px] text-right">{t('roadmap.key_highlights')}</div>
                </div>
              </AnimatedElement>
              {/* Разделительная линия */}
              <div className="w-full h-[1px] bg-[#002D64]" />
              {/* Q1 2025 */}
              <AnimatedElement animation="slideInUp" delay={0}>
                <div className="flex flex-row justify-between items-start gap-0 sm:flex-row sm:gap-8">
                  <div className="text-[#1C1C1C] text-sm sm:text-lg font-normal font-inter min-w-[120px] flex-shrink-0 text-left">{t('roadmap.q1_2025.period')}</div>
                  <div className="max-w-[800px] flex flex-col gap-1 sm:gap-2 text-right">
                    <div className="text-[#1C1C1C] text-sm sm:text-lg font-normal font-inter">• {t('roadmap.q1_2025.item1')}</div>
                    <div className="text-[#1C1C1C] text-sm sm:text-lg font-normal font-inter">• {t('roadmap.q1_2025.item2')}</div>
                    <div className="text-[#1C1C1C] text-sm sm:text-lg font-normal font-inter">• {t('roadmap.q1_2025.item3')}</div>
                    <div className="text-[#1C1C1C] text-sm sm:text-lg font-normal font-inter">• {t('roadmap.q1_2025.item4')}</div>
                  </div>
                </div>
              </AnimatedElement>
              <div className="w-full h-[1px] bg-[#002D64]" />
              {/* Q2 2025 */}
              <AnimatedElement animation="slideInUp" delay={100}>
                <div className="flex flex-row justify-between items-start gap-0 sm:flex-row sm:gap-8">
                  <div className="text-[#1C1C1C] text-sm sm:text-lg font-normal font-inter min-w-[120px] flex-shrink-0 text-left">{t('roadmap.q2_2025.period')}</div>
                  <div className="max-w-[800px] flex flex-col gap-1 sm:gap-2 text-right">
                    <div className="text-[#1C1C1C] text-sm sm:text-lg font-normal font-inter">• {t('roadmap.q2_2025.item1')}</div>
                    <div className="text-[#1C1C1C] text-sm sm:text-lg font-normal font-inter">• {t('roadmap.q2_2025.item2')}</div>
                    <div className="text-[#1C1C1C] text-sm sm:text-lg font-normal font-inter">• {t('roadmap.q2_2025.item3')}</div>
                  </div>
                </div>
              </AnimatedElement>
              <div className="w-full h-[1px] bg-[#002D64]" />
              {/* Q3 2025 */}
              <AnimatedElement animation="slideInUp" delay={200}>
                <div className="flex flex-row justify-between items-start gap-0 sm:flex-row sm:gap-8">
                  <div className="text-[#1C1C1C] text-sm sm:text-lg font-normal font-inter min-w-[120px] flex-shrink-0 text-left">{t('roadmap.q3_2025.period')}</div>
                  <div className="max-w-[800px] flex flex-col gap-1 sm:gap-2 text-right">
                    <div className="text-[#1C1C1C] text-sm sm:text-lg font-normal font-inter">• {t('roadmap.q3_2025.item1')}</div>
                    <div className="text-[#1C1C1C] text-sm sm:text-lg font-normal font-inter">• {t('roadmap.q3_2025.item2')}</div>
                    <div className="text-[#1C1C1C] text-sm sm:text-lg font-normal font-inter">• {t('roadmap.q3_2025.item3')}</div>
                  </div>
                </div>
              </AnimatedElement>
              <div className="w-full h-[1px] bg-[#002D64]" />
              {/* Q4 2025 */}
              <AnimatedElement animation="slideInUp" delay={300}>
                <div className="flex flex-row justify-between items-start gap-0 sm:flex-row sm:gap-8">
                  <div className="text-[#1C1C1C] text-sm sm:text-lg font-normal font-inter min-w-[120px] flex-shrink-0 text-left">{t('roadmap.q4_2025.period')}</div>
                  <div className="max-w-[800px] flex flex-col gap-1 sm:gap-2 text-right">
                    <div className="text-[#1C1C1C] text-sm sm:text-lg font-normal font-inter">• {t('roadmap.q4_2025.item1')}</div>
                    <div className="text-[#1C1C1C] text-sm sm:text-lg font-normal font-inter">• {t('roadmap.q4_2025.item2')}</div>
                    <div className="text-[#1C1C1C] text-sm sm:text-lg font-normal font-inter">• {t('roadmap.q4_2025.item3')}</div>
                  </div>
                </div>
              </AnimatedElement>
              <div className="w-full h-[1px] bg-[#002D64]" />
              {/* 2026+ */}
              <AnimatedElement animation="slideInUp" delay={400}>
                <div className="flex flex-row justify-between items-start gap-0 sm:flex-row sm:gap-8">
                  <div className="text-[#1C1C1C] text-sm sm:text-lg font-normal font-inter min-w-[120px] flex-shrink-0 text-left">{t('roadmap.y2026_plus.period')}</div>
                  <div className="max-w-[800px] flex flex-col gap-1 sm:gap-2 text-right">
                    <div className="text-[#1C1C1C] text-sm sm:text-lg font-normal font-inter">• {t('roadmap.y2026_plus.item1')}</div>
                    <div className="text-[#1C1C1C] text-sm sm:text-lg font-normal font-inter">• {t('roadmap.y2026_plus.item2')}</div>
                    <div className="text-[#1C1C1C] text-sm sm:text-lg font-normal font-inter">• {t('roadmap.y2026_plus.item3')}</div>
                  </div>
                </div>
              </AnimatedElement>
              <div className="w-full h-[1px] bg-[#002D64]" />
            </div>
          </div>
        </section>
        </AnimatedElement>

        {/* Partners (перемещено сюда) */}
        <section id="partners" className="w-full flex flex-col items-center relative py-6 sm:py-8">
          <div className="w-full max-w-2xl sm:max-w-4xl md:max-w-7xl flex flex-col items-center gap-4 sm:gap-8 px-2 sm:px-4">
            <h2 className="text-[#002D64] text-center font-inter text-2xl sm:text-3xl md:text-5xl font-bold leading-tight w-full">{t('partners_title')}</h2>
            <AnimatedPartnersBanner />
          </div>
        </section>

        {/* Tokenomics Section - Full Width */}
        <AnimatedElement animation="slideInUp" delay={400}>
        <section id="tokenomics" className="relative w-full bg-[#EFEFEF] py-8 sm:py-12 md:py-20">
          <div className="w-full h-8 sm:h-[142px] bg-gradient-to-b from-[rgba(217,217,217,0.5)] to-transparent"></div>
          <div className="w-full flex justify-center">
            <div className="sm:max-w-2xl md:max-w-[1190px] w-full px-2 sm:px-4">
              <AnimatedText animation="slideInUp" delay={0}>
                <h2 className="text-2xl sm:text-[32px] md:text-[55.067px] font-normal leading-[120%] mb-6 sm:mb-12 text-center"
                dangerouslySetInnerHTML={{
                  __html: String(t.rich('tokenomics_title', {
                      span: (chunks) => `<span class=\"text-[#002D64] font-bold\">${chunks}</span>`
                  }))
                }}
              />
              </AnimatedText>
              {/* Мобильная версия — горизонтальная карусель карточек */}
              <div className="tokenomics-carousel">
                <div className="flex flex-nowrap gap-4 px-1" style={{ scrollSnapType: 'x mandatory' }}>
                  {/* Public Sale */}
                  <div className="min-w-[85vw] max-w-[90vw] p-4 bg-white rounded-xl shadow flex flex-col gap-2" style={{ scrollSnapAlign: 'start' }}>
                    <div className="font-bold text-[#1C1C1C]">{t('tokenomics_table.category')}</div>
                    <div>{t('tokenomics_table.public_sale.category')}</div>
                    <div className="font-bold text-[#1C1C1C]">{t('tokenomics_table.allocation')}</div>
                    <div>15%</div>
                    <div className="font-bold text-[#1C1C1C]">{t('tokenomics_table.tokens')}</div>
                    <div>150000000</div>
                    <div className="font-bold text-[#1C1C1C]">{t('tokenomics_table.vesting')}</div>
                    <div>{t('tokenomics_table.public_sale.vesting')}</div>
                  </div>
                  {/* Private Rounds */}
                  <div className="min-w-[85vw] max-w-[90vw] p-4 bg-white rounded-xl shadow flex flex-col gap-2" style={{ scrollSnapAlign: 'start' }}>
                    <div className="font-bold text-[#1C1C1C]">{t('tokenomics_table.category')}</div>
                    <div>{t('tokenomics_table.private_rounds.category')}</div>
                    <div className="font-bold text-[#1C1C1C]">{t('tokenomics_table.allocation')}</div>
                    <div>15%</div>
                    <div className="font-bold text-[#1C1C1C]">{t('tokenomics_table.tokens')}</div>
                    <div>150000000</div>
                    <div className="font-bold text-[#1C1C1C]">{t('tokenomics_table.vesting')}</div>
                    <div>{t('tokenomics_table.private_rounds.vesting')}</div>
                  </div>
                  {/* Team & Advisors */}
                  <div className="min-w-[85vw] max-w-[90vw] p-4 bg-white rounded-xl shadow flex flex-col gap-2" style={{ scrollSnapAlign: 'start' }}>
                    <div className="font-bold text-[#1C1C1C]">{t('tokenomics_table.category')}</div>
                    <div>{t('tokenomics_table.team_advisors.category')}</div>
                    <div className="font-bold text-[#1C1C1C]">{t('tokenomics_table.allocation')}</div>
                    <div>15%</div>
                    <div className="font-bold text-[#1C1C1C]">{t('tokenomics_table.tokens')}</div>
                    <div>150000000</div>
                    <div className="font-bold text-[#1C1C1C]">{t('tokenomics_table.vesting')}</div>
                    <div>{t('tokenomics_table.team_advisors.vesting')}</div>
                  </div>
                  {/* Staking Yield */}
                  <div className="min-w-[85vw] max-w-[90vw] p-4 bg-white rounded-xl shadow flex flex-col gap-2" style={{ scrollSnapAlign: 'start' }}>
                    <div className="font-bold text-[#1C1C1C]">{t('tokenomics_table.category')}</div>
                    <div>{t('tokenomics_table.staking_yield.category')}</div>
                    <div className="font-bold text-[#1C1C1C]">{t('tokenomics_table.allocation')}</div>
                    <div>20%</div>
                    <div className="font-bold text-[#1C1C1C]">{t('tokenomics_table.tokens')}</div>
                    <div>200000000</div>
                    <div className="font-bold text-[#1C1C1C]">{t('tokenomics_table.vesting')}</div>
                    <div>{t('tokenomics_table.staking_yield.vesting')}</div>
                  </div>
                  {/* Ecosystem */}
                  <div className="min-w-[85vw] max-w-[90vw] p-4 bg-white rounded-xl shadow flex flex-col gap-2" style={{ scrollSnapAlign: 'start' }}>
                    <div className="font-bold text-[#1C1C1C]">{t('tokenomics_table.category')}</div>
                    <div>{t('tokenomics_table.ecosystem.category')}</div>
                    <div className="font-bold text-[#1C1C1C]">{t('tokenomics_table.allocation')}</div>
                    <div>20%</div>
                    <div className="font-bold text-[#1C1C1C]">{t('tokenomics_table.tokens')}</div>
                    <div>200000000</div>
                    <div className="font-bold text-[#1C1C1C]">{t('tokenomics_table.vesting')}</div>
                    <div dangerouslySetInnerHTML={{ __html: t('tokenomics_table.ecosystem.vesting') }} />
                  </div>
                  {/* DAO Treasury */}
                  <div className="min-w-[85vw] max-w-[90vw] p-4 bg-white rounded-xl shadow flex flex-col gap-2" style={{ scrollSnapAlign: 'start' }}>
                    <div className="font-bold text-[#1C1C1C]">{t('tokenomics_table.category')}</div>
                    <div>{t('tokenomics_table.dao_treasury.category')}</div>
                    <div className="font-bold text-[#1C1C1C]">{t('tokenomics_table.allocation')}</div>
                    <div>10%</div>
                    <div className="font-bold text-[#1C1C1C]">{t('tokenomics_table.tokens')}</div>
                    <div>100000000</div>
                    <div className="font-bold text-[#1C1C1C]">{t('tokenomics_table.vesting')}</div>
                    <div dangerouslySetInnerHTML={{ __html: t('tokenomics_table.dao_treasury.vesting') }} />
                  </div>
                  {/* Liquidity */}
                  <div className="min-w-[85vw] max-w-[90vw] p-4 bg-white rounded-xl shadow flex flex-col gap-2" style={{ scrollSnapAlign: 'start' }}>
                    <div className="font-bold text-[#1C1C1C]">{t('tokenomics_table.category')}</div>
                    <div>{t('tokenomics_table.liquidity.category')}</div>
                    <div className="font-bold text-[#1C1C1C]">{t('tokenomics_table.allocation')}</div>
                    <div>5%</div>
                    <div className="font-bold text-[#1C1C1C]">{t('tokenomics_table.tokens')}</div>
                    <div>50000000</div>
                    <div className="font-bold text-[#1C1C1C]">{t('tokenomics_table.vesting')}</div>
                    <div>{t('tokenomics_table.liquidity.vesting')}</div>
                  </div>
                </div>
              </div>
              {/* Десктопная версия — таблица */}
              <div className="tokenomics-table">
                <div className="text-[#1C1C1C] font-inter text-base sm:text-2xl font-bold justify-self-start">{t('tokenomics_table.category')}</div>
                <div className="text-[#1C1C1C] font-inter text-base sm:text-2xl font-bold text-center">{t('tokenomics_table.allocation')}</div>
                <div className="text-[#1C1C1C] font-inter text-base sm:text-2xl font-bold text-center">{t('tokenomics_table.tokens')}</div>
                <div className="text-[#1C1C1C] font-inter text-base sm:text-2xl font-bold justify-self-start">{t('tokenomics_table.vesting')}</div>
                <div className="col-span-4 h-[1px] bg-[#002D64] my-1 sm:my-2 justify-self-stretch" />
                {/* Строки данных с анимацией */}
                <div className="text-[#1C1C1C] font-inter text-lg font-normal justify-self-start">{t('tokenomics_table.public_sale.category')}</div>
                <AnimatedElement animation="slideInUp" delay={100} className="w-full text-center">
                  <CounterAnimation end={15} duration={1200} suffix="%" className="text-[#1C1C1C] font-inter text-lg font-normal" />
                </AnimatedElement>
                <AnimatedElement animation="slideInUp" delay={150} className="w-full text-center">
                  <CounterAnimation end={150000000} duration={1200} className="text-[#1C1C1C] font-inter text-lg font-normal" />
                </AnimatedElement>
                <div className="text-[#1C1C1C] font-inter text-lg font-normal justify-self-start">{t('tokenomics_table.public_sale.vesting')}</div>
                <div className="text-[#1C1C1C] font-inter text-lg font-normal justify-self-start">{t('tokenomics_table.private_rounds.category')}</div>
                <AnimatedElement animation="slideInUp" delay={200} className="w-full text-center">
                  <CounterAnimation end={15} duration={1200} suffix="%" className="text-[#1C1C1C] font-inter text-lg font-normal" />
                </AnimatedElement>
                <AnimatedElement animation="slideInUp" delay={250} className="w-full text-center">
                  <CounterAnimation end={150000000} duration={1200} className="text-[#1C1C1C] font-inter text-lg font-normal" />
                </AnimatedElement>
                <div className="text-[#1C1C1C] font-inter text-lg font-normal justify-self-start">{t('tokenomics_table.private_rounds.vesting')}</div>
                <div className="text-[#1C1C1C] font-inter text-lg font-normal justify-self-start">{t('tokenomics_table.team_advisors.category')}</div>
                <AnimatedElement animation="slideInUp" delay={300} className="w-full text-center">
                  <CounterAnimation end={15} duration={1200} suffix="%" className="text-[#1C1C1C] font-inter text-lg font-normal" />
                </AnimatedElement>
                <AnimatedElement animation="slideInUp" delay={350} className="w-full text-center">
                  <CounterAnimation end={150000000} duration={1200} className="text-[#1C1C1C] font-inter text-lg font-normal" />
                </AnimatedElement>
                <div className="text-[#1C1C1C] font-inter text-lg font-normal justify-self-start">{t('tokenomics_table.team_advisors.vesting')}</div>
                <div className="text-[#1C1C1C] font-inter text-lg font-normal justify-self-start">{t('tokenomics_table.staking_yield.category')}</div>
                <AnimatedElement animation="slideInUp" delay={400} className="w-full text-center">
                  <CounterAnimation end={20} duration={1200} suffix="%" className="text-[#1C1C1C] font-inter text-lg font-normal" />
                </AnimatedElement>
                <AnimatedElement animation="slideInUp" delay={450} className="w-full text-center">
                  <CounterAnimation end={200000000} duration={1200} className="text-[#1C1C1C] font-inter text-lg font-normal" />
                </AnimatedElement>
                <div className="text-[#1C1C1C] font-inter text-lg font-normal justify-self-start">{t('tokenomics_table.staking_yield.vesting')}</div>
                <div className="text-[#1C1C1C] font-inter text-lg font-normal justify-self-start">{t('tokenomics_table.ecosystem.category')}</div>
                <AnimatedElement animation="slideInUp" delay={500} className="w-full text-center">
                  <CounterAnimation end={20} duration={1200} suffix="%" className="text-[#1C1C1C] font-inter text-lg font-normal" />
                </AnimatedElement>
                <AnimatedElement animation="slideInUp" delay={550} className="w-full text-center">
                  <CounterAnimation end={200000000} duration={1200} className="text-[#1C1C1C] font-inter text-lg font-normal" />
                </AnimatedElement>
                <div className="text-[#1C1C1C] font-inter text-lg font-normal justify-self-start" dangerouslySetInnerHTML={{ __html: t('tokenomics_table.ecosystem.vesting') }} />
                <div className="text-[#1C1C1C] font-inter text-lg font-normal justify-self-start">{t('tokenomics_table.dao_treasury.category')}</div>
                <AnimatedElement animation="slideInUp" delay={600} className="w-full text-center">
                  <CounterAnimation end={10} duration={1200} suffix="%" className="text-[#1C1C1C] font-inter text-lg font-normal" />
                </AnimatedElement>
                <AnimatedElement animation="slideInUp" delay={650} className="w-full text-center">
                  <CounterAnimation end={100000000} duration={1200} className="text-[#1C1C1C] font-inter text-lg font-normal" />
                </AnimatedElement>
                <div className="text-[#1C1C1C] font-inter text-lg font-normal justify-self-start" dangerouslySetInnerHTML={{ __html: t('tokenomics_table.dao_treasury.vesting') }} />
                <div className="text-[#1C1C1C] font-inter text-lg font-normal justify-self-start">{t('tokenomics_table.liquidity.category')}</div>
                <AnimatedElement animation="slideInUp" delay={700} className="w-full text-center">
                  <CounterAnimation end={5} duration={1200} suffix="%" className="text-[#1C1C1C] font-inter text-lg font-normal" />
                </AnimatedElement>
                <AnimatedElement animation="slideInUp" delay={750} className="w-full text-center">
                  <CounterAnimation end={50000000} duration={1200} className="text-[#1C1C1C] font-inter text-lg font-normal" />
                </AnimatedElement>
                <div className="text-[#1C1C1C] font-inter text-lg font-normal justify-self-start">{t('tokenomics_table.liquidity.vesting')}</div>
                </div>
            </div>
          </div>
        </section>
        </AnimatedElement>
        {/* The New Investment Vector Section */}
        <section className="relative w-full bg-white py-8 sm:py-16 flex flex-col items-center">
          <div className="w-full max-w-xs sm:max-w-2xl md:max-w-screen-2xl px-2 sm:px-4 md:px-12 flex flex-col items-center">
            <h2 className="text-[#002D64] text-center font-inter text-xl sm:text-3xl md:text-5xl font-bold leading-tight mb-2 sm:mb-4">{t('new_investment_vector.title')}</h2>
            <p className="text-[#1C1C1C] text-center text-sm sm:text-base md:text-lg max-w-2xl mb-4 sm:mb-6">{t('new_investment_vector.subtitle')}</p>
            <InvestmentVectorCarousel />
            <div 
              className="text-center mt-4 sm:mt-8 text-base sm:text-lg font-normal font-inter leading-relaxed"
              dangerouslySetInnerHTML={{
                __html: String(t.rich('new_investment_vector.description', {
                  span: (chunks) => `<span class=\"text-[#002D64]\">${chunks}</span>`
                }))
              }}
            />
          </div>
        </section>

        {/* What Investors are looking for in 2025 Section */}
        <section className="relative w-full bg-[#EFEFEF] py-8 md:py-12 lg:py-16 relative min-h-[400px] lg:min-h-[584px]">
          {/* Background Image */}
          <div className="absolute inset-0 w-full h-full">
            <Image
              src="/images/lake-token/body/Group 15.svg"
              alt="Background"
              fill
              className="object-cover opacity-10"
            />
          </div>
          
          <div className="w-full max-w-[1728px] mx-auto px-4 md:px-8 lg:px-12 xl:px-[294px] flex flex-col items-center relative">
            <AnimatedText animation="slideInUp" delay={0}>
              <h2 className="text-[#002D64] text-2xl md:text-3xl lg:text-4xl xl:text-[55.067px] font-bold leading-[120%] mb-8 md:mb-10 lg:mb-12 text-center px-4">
                {t('investors_2025.title')}
              </h2>
            </AnimatedText>
            
            <div className="w-full max-w-[1140px] flex flex-col lg:flex-row gap-6 md:gap-8 px-4">
              {/* Left Column */}
              <div className="flex-1 flex flex-col gap-6 md:gap-8">
                {/* Sustainable On-Chain Real Yield */}
                <div className="flex gap-4 md:gap-6 items-start">
                  <div className="w-16 h-16 md:w-[79px] md:h-[79px] flex-shrink-0">
                    <Image
                      src="/images/lake-token/body/Mask group 11.svg"
                      alt="Income Icon"
                      width={79}
                      height={79}
                      className="w-full h-full object-contain"
                    />
                  </div>
                  <div className="flex-1">
                    <AnimatedText animation="fadeIn" delay={100}>
                      <h3 className="text-[#1C1C1C] text-base md:text-lg font-bold mb-2 md:mb-3">
                        {t('investors_2025.asset_backed.title')}
                      </h3>
                    </AnimatedText>
                    <AnimatedText animation="fadeIn" delay={200}>
                      <p className="text-[#1C1C1C] text-sm md:text-lg font-normal leading-relaxed">
                        {t('investors_2025.asset_backed.description')}
                      </p>
                    </AnimatedText>
                  </div>
                </div>

                {/* Infrastructure & Legal Clarity */}
                <div className="flex gap-4 md:gap-6 items-start">
                  <div className="w-16 h-16 md:w-[79px] md:h-[79px] flex-shrink-0">
                    <Image
                      src="/images/lake-token/body/Mask group 22.svg"
                      alt="Validity Icon"
                      width={79}
                      height={79}
                      className="w-full h-full object-contain"
                    />
                  </div>
                  <div className="flex-1">
                    <AnimatedText animation="fadeIn" delay={300}>
                      <h3 className="text-[#1C1C1C] text-base md:text-lg font-bold mb-2 md:mb-3">
                        {t('investors_2025.bridges_finance.title')}
                      </h3>
                    </AnimatedText>
                    <AnimatedText animation="fadeIn" delay={400}>
                      <p className="text-[#1C1C1C] text-sm md:text-lg font-normal leading-relaxed">
                        {t('investors_2025.bridges_finance.description')}
                      </p>
                    </AnimatedText>
                  </div>
                </div>
              </div>

              {/* Right Column */}
              <div className="flex-1 flex flex-col gap-6 md:gap-8">
                {/* Asset-Backed Tokens */}
                <div className="flex gap-4 md:gap-6 items-start">
                  <div className="w-16 h-16 md:w-[79px] md:h-[79px] flex-shrink-0">
                    <Image
                      src="/images/lake-token/body/Mask group 33.svg"
                      alt="Land Icon"
                      width={79}
                      height={79}
                      className="w-full h-full object-contain"
                    />
                  </div>
                  <div className="flex-1">
                    <AnimatedText animation="fadeIn" delay={500}>
                      <h3 className="text-[#1C1C1C] text-base md:text-lg font-bold mb-2 md:mb-3">
                        {t('investors_2025.infrastructure_legal.title')}
                      </h3>
                    </AnimatedText>
                    <AnimatedText animation="fadeIn" delay={600}>
                      <p className="text-[#1C1C1C] text-sm md:text-lg font-normal leading-relaxed">
                        {t('investors_2025.infrastructure_legal.description')}
                      </p>
                    </AnimatedText>
                  </div>
                </div>

                {/* Bridges Between Traditional Finance and Web3 */}
                <div className="flex gap-4 md:gap-6 items-start">
                  <div className="w-16 h-16 md:w-[79px] md:h-[79px] flex-shrink-0">
                    <Image
                      src="/images/lake-token/body/Mask group 44.svg"
                      alt="Digital Key Icon"
                      width={79}
                      height={79}
                      className="w-full h-full object-contain"
                    />
                  </div>
                  <div className="flex-1">
                    <AnimatedText animation="fadeIn" delay={700}>
                      <h3 className="text-[#1C1C1C] text-base md:text-lg font-bold mb-2 md:mb-3">
                        {t('investors_2025.sustainable_yield.title')}
                      </h3>
                    </AnimatedText>
                    <AnimatedText animation="fadeIn" delay={800}>
                      <p className="text-[#1C1C1C] text-sm md:text-lg font-normal leading-relaxed">
                        {t('investors_2025.sustainable_yield.description')}
                      </p>
                    </AnimatedText>
                  </div>
                </div>
              </div>
            </div>
          </div>
        </section>

        {/* Why Now Section */}
        <AnimatedElement animation="slideInUp" delay={500}>
        <section className="relative w-full bg-white flex items-center min-h-[400px] lg:min-h-[832px]">
          <div className="lake-token-section-content w-full max-w-[1728px] mx-auto px-4 md:px-8 lg:px-12 xl:px-[294px] flex flex-col lg:flex-row items-center gap-8 lg:gap-12 h-full">
            <div className="flex-1 flex justify-center items-center h-full">
              <div className="w-full max-w-[534px] aspect-[267/316] overflow-hidden flex items-center h-full">
                <Image 
                  src="/images/lake-token/body/Frame 4 1.svg" 
                  alt="Why Now Illustration" 
                  width={534}
                  height={632}
                  className="w-full h-full object-contain"
                />
              </div>
            </div>
            <div className="flex-1 flex flex-col items-start max-w-[479px] h-full justify-center">
              {/* Group 735 - Заголовки */}
              <div className="w-full mb-8">
                <AnimatedText animation="slideInUp" delay={0}>
                  <h2 className="text-[#002D64] font-inter text-[55.067px] font-bold leading-[120%] mb-4 break-words">{t('why_now.title')}</h2>
                </AnimatedText>
                <AnimatedText animation="fadeIn" delay={100}>
                  <h3 className="text-[#002D64] text-[18px] font-bold leading-normal break-words">{t('why_now.subtitle')}</h3>
                </AnimatedText>
              </div>
              {/* Group 743 - Список */}
              <div className="w-full space-y-4">
                <AnimatedText animation="fadeIn" delay={200}>
                  <div style={{ maxWidth: 331, wordBreak: 'break-word' }}>
                    <span style={{
                      color: '#002D64',
                      fontSize: 18,
                      fontFamily: 'Inter',
                      fontWeight: 400,
                      wordBreak: 'break-word',
                      display: 'block',
                    }}>
                      {t('why_now.overheated_markets')}
                    </span>
                    <span style={{
                      color: '#1C1C1C',
                      fontSize: 18,
                      fontFamily: 'Inter',
                      fontWeight: 400,
                      wordBreak: 'break-word',
                      display: 'block',
                    }}>
                      {t('why_now.investors_seek')}
                    </span>
                  </div>
                </AnimatedText>
                <AnimatedText animation="fadeIn" delay={300}>
                  <p className="text-[#1C1C1C] text-[18px] font-normal leading-normal break-words">
                    {t('why_now.regulatory_pressure')}
                  </p>
                </AnimatedText>
                <AnimatedText animation="fadeIn" delay={400}>
                  <p className="text-[18px] font-normal leading-normal break-words">
                    <span style={{ color: '#1C1C1C' }}>{t('why_now.institutional_entry')} </span>
                    <span style={{ color: '#002D64' }}>{t('why_now.institutional_companies')}</span>
                  </p>
                </AnimatedText>
                <AnimatedText animation="fadeIn" delay={500}>
                  <p className="text-[18px] font-normal leading-normal break-words">
                    <span style={{ color: '#1C1C1C' }}>{t('why_now.investment_amount')} </span>
                    <span style={{ color: '#002D64' }}>{t('why_now.investment_period')}</span>
                  </p>
                </AnimatedText>
              </div>
            </div>
          </div>
        </section>
        </AnimatedElement>

        {/* Buy Lake Token Section — DESKTOP */}
        <section id="buy_lake_token" className="lake-buy-desktop flex relative w-full min-h-[386px] h-[386px] bg-[#EFEFEF] items-center justify-end overflow-hidden">
          {/* Фон */}
          <Image
            src="/images/lake-token/body/Group 84.svg"
            alt="Background"
            width={783}
            height={783}
            className="absolute left-[-33px] top-[-190px] w-[45vw] max-w-[783px] min-w-[180px] h-[45vw] max-h-[783px] min-h-[180px] z-0 select-none pointer-events-none"
            priority
          />
          {/* Контент */}
          <div className="relative z-10 flex flex-col items-end w-full max-w-[900px] pr-10">
            <div className="buy-title w-[684px] text-right mb-1 font-inter" dangerouslySetInnerHTML={{ __html: String(t.raw('buy_lake_title')) }} />
            <a
              href="https://bscscan.com/token/0xa5706783e577b1a9a9a89502efb5387fedaec6e0"
              className="buy-btn flex items-center justify-center mt-2"
              style={{ width: 463, height: 80, background: '#002D64', borderRadius: 16, position: 'relative' }}
              target="_blank"
              rel="noopener noreferrer"
            >
              <div style={{ width: 32, height: 32, display: 'flex', alignItems: 'center', justifyContent: 'center', flexShrink: 0, marginRight: 16 }}>
                <svg width="34" height="34" viewBox="0 0 34 34" fill="none" xmlns="http://www.w3.org/2000/svg">
                  <circle cx="17" cy="17" r="17" fill="white"/>
                  <path d="M11.495 8.60126C11.495 8.60126 10.5198 6.78627 9.95093 6.48828H17.1025L15.4772 8.60126V13.071H17.834V14.6151H15.4772V16.3218H17.834V18.2722H15.4772V23.5546C18.2403 23.7984 19.2155 22.5794 22.5475 20.629C25.2131 19.0686 26.9089 20.2497 27.4236 21.0353C25.3432 21.3604 24.7689 24.8549 24.7418 26.5616H9.78839C11.2187 25.7814 11.5221 23.8526 11.495 22.9858V18.0284H9.30078V16.3218H11.495V14.6151H9.30078V13.071H11.495V8.60126Z" fill="#002D64"/>
                </svg>
              </div>
              <span style={{ color: 'white', fontSize: 24, fontFamily: 'Inter', fontWeight: 700, lineHeight: '28px', wordWrap: 'break-word', flexShrink: 0 }}>Buy</span>
            </a>
          </div>
        </section>
        {/* Buy Lake Token Section — MOBILE */}
        <section id="buy_lake_token_mobile" className="lake-buy-mobile flex flex-col w-full bg-[#EFEFEF] items-center justify-center py-8 px-4 relative overflow-hidden">
          {/* Фон */}
          {/* Контент */}
          <div className="relative z-10 flex flex-col items-center w-full">
            <div
              className="buy-title-mobile text-center mb-6"
              dangerouslySetInnerHTML={{ __html: String(t.raw('buy_lake_title')) }}
              style={{ fontSize: 'clamp(1.3rem, 6vw, 2rem)', lineHeight: '1.2', fontWeight: 700, fontFamily: 'Inter', color: '#002D64' }}
            />
            <a
              href="https://bscscan.com/token/0xa5706783e577b1a9a9a89502efb5387fedaec6e0"
              className="buy-btn-mobile flex items-center justify-center mt-2"
              style={{ width: '90vw', maxWidth: 320, height: 56, borderRadius: 12, background: '#002D64', position: 'relative' }}
              target="_blank"
              rel="noopener noreferrer"
            >
              <div style={{ width: 24, height: 24, display: 'flex', alignItems: 'center', justifyContent: 'center', flexShrink: 0, marginRight: 12 }}>
                <svg width="28" height="28" viewBox="0 0 34 34" fill="none" xmlns="http://www.w3.org/2000/svg">
                  <circle cx="17" cy="17" r="17" fill="white"/>
                  <path d="M11.495 8.60126C11.495 8.60126 10.5198 6.78627 9.95093 6.48828H17.1025L15.4772 8.60126V13.071H17.834V14.6151H15.4772V16.3218H17.834V18.2722H15.4772V23.5546C18.2403 23.7984 19.2155 22.5794 22.5475 20.629C25.2131 19.0686 26.9089 20.2497 27.4236 21.0353C25.3432 21.3604 24.7689 24.8549 24.7418 26.5616H9.78839C11.2187 25.7814 11.5221 23.8526 11.495 22.9858V18.0284H9.30078V16.3218H11.495V14.6151H9.30078V13.071H11.495V8.60126Z" fill="#002D64"/>
                </svg>
              </div>
              <span style={{ color: 'white', fontSize: 18, fontFamily: 'Inter', fontWeight: 700, lineHeight: '22px', wordWrap: 'break-word', flexShrink: 0 }}>Buy</span>
            </a>
          </div>
        </section>

        {/* Footer (Contact us + Resources) */}
        <AnimatedElement animation="fadeIn" delay={600}>
          <LakeTokenFooter />
        </AnimatedElement>
      </div>
    </div>
  );
}